/*
 * ManagedLDAPConnection.java
 *
 * Created on 9. Januar 2005, 04:15
 */
package de.alexanderlindhorst.jca.ldap;

import org.apache.log4j.*;

import java.io.*;

import java.util.*;

import javax.naming.*;
import javax.naming.ldap.*;

import javax.resource.*;
import javax.resource.spi.*;


/**
 * The managed connection to the LDAP EIS that also provides the application
 * handle connection.
 * @author  $Author: lindhrst $
 * @version $Revision: 1.2 $
 */
public class ManagedLDAPConnection implements ManagedConnection,
    LDAPConnectorConstants {
    private static Logger logger = Logger.getLogger(LOG_CATEGORY);
    private static final String DRIVER = "com.sun.jndi.ldap.LdapCtxFactory";
    private PrintWriter logWriter = null;
    private HashSet listeners = null;

    //keep track of active handles
    private Stack handles = null;
    private LDAPConnectionInfo settings = null;
    private LdapContext ctx = null;
    private boolean valid = true;
    private LDAPConnectionMetaData meta = null;

    public ManagedLDAPConnection(LDAPConnectionInfo settings)
        throws ResourceException {
        //initialize state
        listeners = new HashSet(10, .8f);
        handles = new Stack();
        this.settings = settings;

        //build up connection physically
        Hashtable env = new Hashtable();
        String value = null;

        //1. the host
        value = "ldap://" + settings.getHost() + ":389";
        env.put(Context.PROVIDER_URL, value);

        //2. the driver
        env.put(Context.INITIAL_CONTEXT_FACTORY, DRIVER);

        //3. connection principal
        value = settings.getPrincipal();
        env.put(Context.SECURITY_PRINCIPAL, value);

        //4. connection principal's credentials
        value = settings.getCredentials();
        env.put(Context.SECURITY_CREDENTIALS, value);

        //5. automagic pooling
        env.put("com.sun.jndi.ldap.connect.pool", "true");

        //6. timeout = 2 secs
        env.put("com.sun.jndi.ldap.connect.timeout", "2000");

        //derive connection object
        logger.debug(getClass().toString() +
            " - About to retrieve new physical connection");

        try {
            ctx = new InitialLdapContext(env, null);
            logger.info(getClass().toString() +
                " - Got a new connection with " + settings.toString());
            valid = true;
            meta = new LDAPConnectionMetaData(settings.getPrincipal());
        } catch (Exception e) {
            valid = false;
            throw new ResourceException(getClass().toString() +
                " - Couldn't create LDAP context", e);
        }
    }

    public void addConnectionEventListener(
        javax.resource.spi.ConnectionEventListener connectionEventListener) {
        if (!valid) {
            throw new RuntimeException("Not a valid physical connection.");
        }

        if (connectionEventListener == null) {
            return;
        }

        listeners.add(connectionEventListener);
    }

    public void associateConnection(Object obj)
        throws javax.resource.ResourceException {
        if (!valid) {
            throw new LDAPEISUnavailableException(
                "Not a valid physical connection.");
        }

        LDAPConnection c = (LDAPConnection) obj;

        if (c != null) {
            c.setAssociatedManagedConnection(this);
        }

        logger.debug(getClass().toString() + " - Connection association with " +
            c);
    }

    public void cleanup() throws javax.resource.ResourceException {
        //nuttn to do?
        logger.debug(getClass() + " - cleanup requested");

        //invalidate all handles
        LDAPConnection c = null;

        while (!handles.isEmpty()) {
            c = (LDAPConnection) handles.pop();
            c.setActive(false);
        }

        logger.debug(getClass() + " - Cleanup done");
    }

    public void destroy() throws javax.resource.ResourceException {
        //nuttn to do?
        logger.debug(getClass() + " - destruction requested");

        cleanup();

        try {
            ctx.close();
        } catch (Exception e) {
            logger.fatal("Couldn't close context: " + e.getMessage());
        }

        logger.debug(getClass() + " - destruction done");
    }

    public Object getConnection(javax.security.auth.Subject subject,
        javax.resource.spi.ConnectionRequestInfo connectionRequestInfo)
        throws javax.resource.ResourceException {
        if (!valid) {
            throw new LDAPEISUnavailableException(
                "Not a valid physical connection.");
        }

        logger.debug("About to retrieve new handle; request info already " +
            "configured, will be ignored");

        LDAPConnection conn = new LDAPConnection(ctx, this);

        // done automatically: associateConnection(conn);
        //keep track of active handle
        if (!handles.isEmpty()) {
            ((LDAPConnection) handles.peek()).setActive(false);
        }

        handles.push(conn);
        conn.setActive(true);

        logger.debug("handle retrieved: " + conn);

        return conn;
    }

    public javax.resource.spi.LocalTransaction getLocalTransaction()
        throws javax.resource.ResourceException {
        logger.warn("Warning: Transactions are not supported.");

        return null;
    }

    public java.io.PrintWriter getLogWriter()
        throws javax.resource.ResourceException {
        return logWriter;
    }

    public javax.resource.spi.ManagedConnectionMetaData getMetaData()
        throws javax.resource.ResourceException {
        return meta;
    }

    public javax.transaction.xa.XAResource getXAResource()
        throws javax.resource.ResourceException {
        logger.error("Transactions are not supported, no such object.");
        throw new UnsupportedOperationException();
    }

    public void removeConnectionEventListener(
        javax.resource.spi.ConnectionEventListener connectionEventListener) {
        if (connectionEventListener == null) {
            return;
        }

        listeners.remove(connectionEventListener);
    }

    public void setLogWriter(java.io.PrintWriter printWriter)
        throws javax.resource.ResourceException {
        if (!valid) {
            throw new LDAPEISUnavailableException(
                "Not a valid physical connection.");
        }

        logWriter = printWriter;
    }

    private void fireConnectionEvent(ConnectionEvent event) {
        Iterator iterator = listeners.iterator();

        switch (event.getId()) {
        case ConnectionEvent.CONNECTION_CLOSED:

            while (iterator.hasNext()) {
                ((ConnectionEventListener) iterator.next()).connectionClosed(event);
            }

            break;

        case ConnectionEvent.CONNECTION_ERROR_OCCURRED:

            while (iterator.hasNext()) {
                ((ConnectionEventListener) iterator.next()).connectionErrorOccurred(event);
            }

            break;

        default:
            //do nothing
        }
    }

    boolean isBeingUsed() {
        return !handles.isEmpty();
    }

    ConnectionRequestInfo getConnectionRequestInfo() {
        return settings;
    }

    void setHandleClosed(LDAPConnection connection) {
        //order is important
        logger.debug(getClass() + "Closing handle " + connection);

        synchronized (handles) {
            if (handles.isEmpty()) {
                logger.fatal("Stack is empty but a request for closing a " +
                    "connection was made: " + connection +
                    " -> Buggy Pool Manager?");

                return;
            }

            if (handles.peek().equals(connection)) {
                //notify listeners
                ConnectionEvent ce = new ConnectionEvent(this,
                        ConnectionEvent.CONNECTION_CLOSED);
                ce.setConnectionHandle(connection);
                fireConnectionEvent(ce);

                //do your own cleanup
                /*
                 *Watch it: If event handling is synchronous, the appserver's
                 *handling routine might already have been here and have removed
                 *the connection. Thus we have to check for empty stacks
                 */
                connection.setActive(false);
                if (!handles.isEmpty()) {
                    handles.pop(); //throw it off the stack
                }

                //activate remaining handles
                if (!handles.isEmpty()) {
                    LDAPConnection current = (LDAPConnection) handles.peek();
                    current.setActive(true);
                }

                logger.debug(getClass() + "Handle closed: " + connection);

                return;
            }
        }

        //Everything else is wrong, bad housekeeping etc. Clean it up
        logger.fatal(getClass() +
            "Connection handles returned in wrong order -> shooting myself.");
        fireConnectionEvent(new ConnectionEvent(this,
                ConnectionEvent.CONNECTION_ERROR_OCCURRED));
    }
}
